"""
Enhanced Glyph Attractor Analysis
For pre-existing 3D point cloud glyphs (.png snapshots + .obj models)
"""

import numpy as np
from PIL import Image
import json
from pathlib import Path
from typing import Dict, List, Tuple
import re

class GlyphPointCloudAnalyzer:
    def __init__(self, 
                 rotation_points: int = 26,
                 samples_per_percent: int = 10,
                 deterministic_seed: int = 12345):
        
        self.rotation_points = rotation_points
        self.samples_per_percent = samples_per_percent
        self.seed = deterministic_seed
        
    def load_glyph_from_png(self, png_path: str) -> Dict:
        """
        Extract point cloud from PNG snapshot
        Each colored pixel = one point in original 3D space
        """
        img = Image.open(png_path).convert('RGB')
        arr = np.array(img)
        h, w, _ = arr.shape
        
        points = []
        for y in range(h):
            for x in range(w):
                r, g, b = arr[y, x]
                
                # Only non-black pixels
                if r > 5 or g > 5 or b > 5:
                    points.append({
                        'x': x - w/2,  # Center horizontally
                        'y': -(y - h/2),  # Invert Y (image coords to cartesian)
                        'r': r / 255.0,
                        'g': g / 255.0,
                        'b': b / 255.0,
                        'color_class': self.classify_color(r, g, b)
                    })
        
        return {
            'points': points,
            'width': w,
            'height': h,
            'total_points': len(points)
        }
    
    def classify_color(self, r: int, g: int, b: int) -> str:
        """Classify pixel by dominant color for region analysis"""
        if r > 150 and b > 150 and g < 100:
            return 'purple'  # Top/horns
        elif r > 200 and g > 200 and b > 200:
            return 'white'   # Core/center
        elif g > 150 and r < 100 and b < 100:
            return 'green'   # Bottom/legs
        elif r > 200 and g > 200 and b < 100:
            return 'yellow'  # Mid-body
        else:
            return 'other'
    
    def analyze_original_structure(self, glyph_data: Dict) -> Dict:
        """
        Analyze the original 3D glyph structure
        Before cylindrical transformation
        """
        points = glyph_data['points']
        
        if len(points) == 0:
            return {'error': 'No points found'}
        
        # Spatial symmetry
        left_points = [p for p in points if p['x'] < 0]
        right_points = [p for p in points if p['x'] > 0]
        
        horizontal_sym = min(len(left_points), len(right_points)) / max(len(left_points), len(right_points)) if max(len(left_points), len(right_points)) > 0 else 0
        
        # Vertical distribution
        ys = [p['y'] for p in points]
        y_min, y_max = min(ys), max(ys)
        y_range = y_max - y_min
        
        # Color region distribution
        color_dist = {}
        for p in points:
            c = p['color_class']
            color_dist[c] = color_dist.get(c, 0) + 1
        
        # Calculate center of mass
        com_x = np.mean([p['x'] for p in points])
        com_y = np.mean([p['y'] for p in points])
        
        return {
            'horizontal_symmetry': float(horizontal_sym),
            'vertical_range': float(y_range),
            'center_of_mass': {'x': float(com_x), 'y': float(com_y)},
            'color_distribution': color_dist,
            'spatial_density': self.calculate_spatial_density(points)
        }
    
    def calculate_spatial_density(self, points: List[Dict]) -> Dict:
        """Analyze how tightly packed the points are"""
        if len(points) < 2:
            return {'mean_spacing': 0, 'density_score': 0}
        
        # Sample random pairs to estimate spacing
        np.random.seed(42)
        n_samples = min(100, len(points))
        sampled = np.random.choice(len(points), n_samples, replace=False)
        
        distances = []
        for i in range(len(sampled)-1):
            p1 = points[sampled[i]]
            p2 = points[sampled[i+1]]
            dist = np.sqrt((p1['x'] - p2['x'])**2 + (p1['y'] - p2['y'])**2)
            distances.append(dist)
        
        return {
            'mean_spacing': float(np.mean(distances)),
            'density_score': 1.0 / (1.0 + np.mean(distances))  # Higher = denser
        }
    
    def cylindrical_transform(self, points: List[Dict]) -> Dict:
        """
        Apply the cylindrical transformation to point cloud
        Each 2D point → ring of 3D points
        """
        positions_3d = []
        
        for point in points:
            radius = abs(point['x'])  # Distance from center = radius
            height = point['y']
            
            # Create ring of points at this radius/height
            for rot_idx in range(self.rotation_points):
                angle = (rot_idx / self.rotation_points) * 2 * np.pi
                
                x_3d = radius * np.cos(angle)
                z_3d = radius * np.sin(angle)
                y_3d = height
                
                positions_3d.append({
                    'x': x_3d,
                    'y': y_3d,
                    'z': z_3d,
                    'source_point': point
                })
        
        return {
            'positions': positions_3d,
            'point_count': len(positions_3d),
            'original_count': len(points)
        }
    
    def calculate_3d_symmetry(self, positions: List[Dict]) -> Dict:
        """
        Calculate symmetry metrics in 3D cylindrical space
        """
        # Quadrant analysis (XZ plane)
        quadrants = [0, 0, 0, 0]
        for p in positions:
            x, z = p['x'], p['z']
            if x >= 0 and z >= 0: quadrants[0] += 1
            elif x < 0 and z >= 0: quadrants[1] += 1
            elif x < 0 and z < 0: quadrants[2] += 1
            else: quadrants[3] += 1
        
        total = sum(quadrants)
        quadrant_sym = 1 - (max(quadrants) - min(quadrants)) / total if total > 0 else 0
        
        # Rotational symmetry (angular distribution)
        angles = [np.arctan2(p['z'], p['x']) for p in positions]
        angle_bins = np.histogram(angles, bins=8)[0]
        rotational_sym = 1 - np.std(angle_bins) / (np.mean(angle_bins) + 1e-6)
        
        # Reflective symmetry (X-axis mirror)
        x_positive = sum(1 for p in positions if p['x'] >= 0)
        x_negative = sum(1 for p in positions if p['x'] < 0)
        reflective_sym = min(x_positive, x_negative) / max(x_positive, x_negative) if max(x_positive, x_negative) > 0 else 0
        
        overall = (quadrant_sym + rotational_sym + reflective_sym) / 3
        
        return {
            'quadrant': float(quadrant_sym),
            'rotational': float(rotational_sym),
            'reflective': float(reflective_sym),
            'overall_symmetry': float(overall)
        }
    
    def generate_state_bank(self, 
                           glyph_data: Dict,
                           range_min: int = 1,
                           range_max: int = 10) -> Dict:
        """
        Generate state bank by sampling original points
        Then apply cylindrical transform to each sample
        """
        points = glyph_data['points']
        banks = {}
        
        for percent in range(range_min, range_max + 1):
            banks[percent] = []
            sample_size = max(1, int(len(points) * (percent / 100.0)))
            
            for sample_idx in range(self.samples_per_percent):
                # Deterministic sampling
                np.random.seed(self.seed + percent * 1000 + sample_idx)
                
                if sample_size >= len(points):
                    sampled_points = points
                else:
                    sampled_indices = np.random.choice(
                        len(points), 
                        sample_size, 
                        replace=False
                    )
                    sampled_points = [points[i] for i in sampled_indices]
                
                # Transform to cylinder
                cylindrical_state = self.cylindrical_transform(sampled_points)
                
                # Calculate 3D symmetry
                symmetry = self.calculate_3d_symmetry(cylindrical_state['positions'])
                
                banks[percent].append({
                    'sample_index': sample_idx,
                    'sampled_count': len(sampled_points),
                    'cylinder_count': cylindrical_state['point_count'],
                    'symmetry': symmetry
                })
        
        return self.analyze_bank_evolution(banks)
    
    def analyze_bank_evolution(self, banks: Dict) -> Dict:
        """Analyze how symmetry evolves across sampling percentages"""
        
        evolution = []
        for percent in sorted(banks.keys()):
            avg_symmetry = np.mean([
                s['symmetry']['overall_symmetry'] 
                for s in banks[percent]
            ])
            
            evolution.append({
                'percent': percent,
                'avg_symmetry': float(avg_symmetry),
                'std_symmetry': float(np.std([
                    s['symmetry']['overall_symmetry'] 
                    for s in banks[percent]
                ]))
            })
        
        # Find critical points
        critical_points = []
        for i in range(1, len(evolution)):
            prev_sym = evolution[i-1]['avg_symmetry']
            curr_sym = evolution[i]['avg_symmetry']
            change = abs(curr_sym - prev_sym)
            
            if change > 0.05:
                critical_points.append({
                    'percent': evolution[i]['percent'],
                    'change': float(change),
                    'type': 'increase' if curr_sym > prev_sym else 'decrease'
                })
        
        # Calculate attractor signature
        late_symmetries = [e['avg_symmetry'] for e in evolution[-5:]]
        convergence = np.mean(late_symmetries)
        
        return {
            'evolution': evolution,
            'critical_points': critical_points,
            'convergence_value': float(convergence),
            'convergence_stability': float(np.std(late_symmetries)),
            'attractor_type': self.classify_attractor(convergence)
        }
    
    def classify_attractor(self, convergence: float) -> str:
        """Classify by convergence value"""
        if abs(convergence - 0.333) < 0.02:
            return "THIRD_ATTRACTOR"
        elif abs(convergence - 0.25) < 0.02:
            return "QUARTER_ATTRACTOR"
        elif abs(convergence - 0.5) < 0.02:
            return "HALF_ATTRACTOR"
        elif convergence > 0.7:
            return "HIGH_SYMMETRY"
        elif convergence < 0.2:
            return "CHAOTIC"
        else:
            return "COMPLEX"
    
    def process_glyph(self, png_path: str) -> Dict:
        """Main processing pipeline for one glyph"""
        
        glyph_id = Path(png_path).stem
        
        # Load point cloud from PNG
        glyph_data = self.load_glyph_from_png(png_path)
        
        if 'error' in glyph_data or glyph_data['total_points'] == 0:
            return {
                'glyph_id': glyph_id,
                'error': 'No valid points found'
            }
        
        # Analyze original structure
        original = self.analyze_original_structure(glyph_data)
        
        # Generate state bank (cylindrical transformation analysis)
        bank_results = self.generate_state_bank(glyph_data)
        
        return {
            'glyph_id': glyph_id,
            'path': str(png_path),
            'original_structure': original,
            'total_points': glyph_data['total_points'],
            'bank_analysis': bank_results,
            'attractor_signature': {
                'convergence': bank_results['convergence_value'],
                'stability': bank_results['convergence_stability'],
                'type': bank_results['attractor_type'],
                'critical_points': len(bank_results['critical_points'])
            }
        }


def batch_process_directory(input_dir: str, output_file: str):
    """Process all PNG glyphs in directory"""
    
    analyzer = GlyphPointCloudAnalyzer()
    
    glyph_files = list(Path(input_dir).glob("glyph_snapshot_*.png"))
    print(f"Found {len(glyph_files)} glyph snapshots")
    
    results = []
    for i, png_path in enumerate(glyph_files):
        print(f"Processing {i+1}/{len(glyph_files)}: {png_path.name}")
        
        result = analyzer.process_glyph(str(png_path))
        results.append(result)
        
        # Save incrementally every 100
        if (i + 1) % 100 == 0:
            with open(output_file, 'w') as f:
                json.dump({'glyphs': results}, f, indent=2)
            print(f"  → Saved {i+1} results")
    
    # Final save
    with open(output_file, 'w') as f:
        json.dump({
            'total_glyphs': len(results),
            'glyphs': results,
            'metadata': {
                'rotation_points': analyzer.rotation_points,
                'seed': analyzer.seed
            }
        }, f, indent=2)
    
    print(f"\n✓ Complete! Results in {output_file}")
    
    # Quick analysis
    analyze_results(results)
    
    return results


def analyze_results(results: List[Dict]):
    """Quick statistical summary"""
    
    valid = [r for r in results if 'error' not in r]
    
    print(f"\n=== ANALYSIS SUMMARY ===")
    print(f"Valid glyphs: {len(valid)}/{len(results)}")
    
    # Attractor distribution
    by_type = {}
    for r in valid:
        atype = r['attractor_signature']['type']
        by_type[atype] = by_type.get(atype, 0) + 1
    
    print("\nAttractor Distribution:")
    for atype, count in sorted(by_type.items(), key=lambda x: -x[1]):
        pct = count / len(valid) * 100
        print(f"  {atype}: {count} ({pct:.1f}%)")
    
    # Convergence statistics
    convergences = [r['attractor_signature']['convergence'] for r in valid]
    print(f"\nConvergence Values:")
    print(f"  Mean: {np.mean(convergences):.4f}")
    print(f"  Median: {np.median(convergences):.4f}")
    print(f"  Std: {np.std(convergences):.4f}")
    print(f"  Range: {min(convergences):.4f} - {max(convergences):.4f}")
    
    # Find interesting cases
    print(f"\nExtreme Cases:")
    highest = max(valid, key=lambda r: r['attractor_signature']['convergence'])
    print(f"  Highest: {highest['glyph_id']} = {highest['attractor_signature']['convergence']:.4f}")
    
    lowest = min(valid, key=lambda r: r['attractor_signature']['convergence'])
    print(f"  Lowest: {lowest['glyph_id']} = {lowest['attractor_signature']['convergence']:.4f}")


if __name__ == "__main__":
    import sys
    
    if len(sys.argv) < 2:
        print("Usage: python glyph_analyzer.py <glyph_directory>")
        sys.exit(1)
    
    glyph_dir = sys.argv[1]
    results = batch_process_directory(glyph_dir, "glyph_analysis_results.json")